/* 
   CC0 2011, Martin Haye

   To the extent possible under law, Martin Haye has waived all copyright 
   and related or neighboring rights to p2e: Pseudo-II Emulator. 
   This work is published from: United States.
*/

/* ======================================================================= */

function AssertException(message) { this.message = message; }
AssertException.prototype.toString = function () {
  return 'AssertException: ' + this.message;
}

function assert(exp, message) {
  if (!exp) {
    throw new AssertException(message);
  }
}

/* ======================================================================= */

// Enable checking?
var checker = null; //new Checker();
var last_run_t = 0;

var ballsOut = false;
var rebootInterval = -1; //ballsOut ? 100 * 1000000 : -1;
var rebootAfterTime = rebootInterval;

var softCapsLock = true;

/* ======================================================================= */

// Saveable state
var raw_mem;
var mem_set;
var mem_run;
var blockMap;
var volatile1, volatile2;
var lastVolatilityFlush = 0;
var hibank_get;

var mem_get_saved = null; // Used to shunt when SETAUXRD is on
var mem_set_saved = []; // Used to shunt when SETAUXWR is on

var bankState;

var CxxxMode;
var DxxxMode;
var ExxxMode;
var lcWriteMode = "protect";

// Primary display switches
var displayPage = 0;          // 0 or 1
var textSwitch = "text";      // "text" or "graphics"
var mixedSwitch = "unmixed";  // "unmixed" or "mixed"
var hgrSwitch = "gr";         // "gr" or "hgr"

// Derived display characteristics
var upperMode = "text"; // "text", "gr", or "hgr": main part of screen
var lowerMode = "text"; // "text", "gr", or "hgr": bottom of screen

var diskDrive;

var kbdBuf = "";
var prevKey = 0;
var keyIsDown = false;

// Non-saveable state
var rom_data;
var kbdPause = 0;
var kbdPauseConst = 15;

function def_run() 
{
  branched = false;
  while (!branched)
    opTbl[raw_mem[pc++]]();
}

var volatileCounts = []; // only populate when needed

function volatile_run() 
{
  opTbl[raw_mem[pc++]](); // just one, hopefully next can be compiled
}

function mem_get(addr) {
  if (addr < 0xC000)
    return raw_mem[addr];
  else
    return hibank_get[addr>>12][addr](addr);
}

/** Read from address in the I/O space */
function mem_get_C0xx(addr) 
{
  switch (addr) {
    case 0xC000: // read keyboard
      if (kbdBuf != "") {
        prevKey = (kbdBuf.charCodeAt(0) & 0xFF) | 0x80;
        kbdBuf = kbdBuf.substr(1);
      }
      return prevKey;
  
    case 0xC010: // strobe keyboard
      if (prevKey < 0x80 && kbdBuf != "") {
        prevKey = (kbdBuf.charCodeAt(0) & 0xFF) | 0x80;
        kbdBuf = kbdBuf.substr(1);
      }
      else
        prevKey &= 0x7F;
      return (prevKey & 0x7F) | (keyIsDown ? 0x80 : 0);
      
    case 0xC011: // reading from LC bank 2?
      return DxxxMode == "ram2" ? 128 : 0;
      
    case 0xC012: // reading from LC ram?
      return lcReadMode == "ram" ? 128 : 0;
      
    case 0xC015: // RDCXROM
      return CxxxMode == "ROM" ? 128 : 0;
      
    case 0xC01A: // RDTEXT
      return upperMode == "text" ? 128 : 0;
  
    case 0xC01B: // RDMIXED
      return upperMode != lowerMode ? 128 : 0;
      
    case 0xC01C: // RDPAGE2
      return displayPage ? 128 : 0;
      
    case 0xC01D: // RDHIRES
      return upperMode == "hgr" ? 128 : 0;
      
    case 0xC050: // CLRTEXT
      textSwitch = "graphics";
      updateDisplayMode();
      break;
    
    case 0xC051: // SETTEXT
      textSwitch = "text";
      updateDisplayMode();
      break;
    
    case 0xC052: // CLRMIXED
      mixedSwitch = "unmixed";
      updateDisplayMode();
      break;
    
    case 0xC053: // SETMIXED
      mixedSwitch = "mixed";
      updateDisplayMode();
      break;
    
    case 0xC054: // PAGE1
      switchDisplayPage(0);
      break;
    
    case 0xC055: // PAGE2
      switchDisplayPage(1);
      break;
    
    case 0xC056: // CLRHIRES
      hgrSwitch = "gr";
      updateDisplayMode();
      break;
    
    case 0xC057: // SETHIRES
      hgrSwitch = "hgr";
      updateDisplayMode();
      break;

    case 0xC080:
    case 0xC084:
      switchDxxxMode("ram2");
      switchExxxMode("ram");
      lcWriteMode = "protect";
      return 0;
      
    case 0xC081:
    case 0xC085:
      switchDxxxMode("rom");
      switchExxxMode("rom");
      lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write2" : "protect2";
      return 0;
      
    case 0xC082:
    case 0xC086:
      switchDxxxMode("rom");
      switchExxxMode("rom");
      lcWriteMode = "protect";
      return 0;
      
    case 0xC083:
    case 0xC087:
      switchDxxxMode("ram2");
      switchExxxMode("ram");
      lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write2" : "protect2";
      return 0;
      
    case 0xC088:
    case 0xC08C:
      switchDxxxMode("ram1");
      switchExxxMode("ram");
      lcWriteMode = "protect";
      return 0;
      
    case 0xC089:
    case 0xC08D:
      switchDxxxMode("rom");
      switchExxxMode("rom");
      lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write1" : "protect2";
      return 0;
      
    case 0xC08A:
    case 0xC08E:
      switchDxxxMode("rom");
      switchExxxMode("rom");
      lcWriteMode = "protect";
      return 0;
      
    case 0xC08B:
    case 0xC08F:
      switchDxxxMode("ram1");
      switchExxxMode("ram");
      lcWriteMode = (lcWriteMode=="protect2" || lcWriteMode == "write1" || lcWriteMode=="write2") ? "write1" : "protect2";
      return 0;
      
    default:
      if (addr >= 0xC0E0 && addr <= 0xC0EF) // Disk II
        return diskDrive.softswitch_get(addr);
  
      // No match, zero seems a safe general return value.
      return 0;
  }
}

function updateDisplayMode()
{
  var prevUpperMode = upperMode;
  var prevLowerMode = lowerMode;
  
  if (textSwitch == "text")
    upperMode = lowerMode = "text";
  else if (hgrSwitch == "hgr") {
    upperMode = "hgr";
    lowerMode = mixedSwitch == "mixed" ? text : "hgr";
  }
  else {
    upperMode = "gr";
    lowerMode = mixedSwitch == "mixed" ? text : "gr";
  }
  
  // No change? Do nothing.
  if (upperMode == prevUpperMode && lowerMode == prevLowerMode)
    return;
    
  // Update the setters
  var i;
  for (i = 0x400; i < 0xC00; i++)
    mem_set[i] = calc_setter(i);
  for (i = 0x2000; i < 0x6000; i++)
    mem_set[i] = calc_setter(i);
  
  // And redraw the screen
  if (textSwitch == "text")
    redrawText();
  else
    redrawHires();
}

function switchDisplayPage(page)
{
  //console.debug("display page (currently no-op): " + page);
}

/*
  Notes on structure of ROM files
  ===============================
  
  APPLE2+.ROM: 20 KB
  
    0000.0FFF: C000.CFFF slot ROMs
    1000.1FFF: repeat of slot ROMs
    2000.4FFF: D000.FFFF Applesoft and Monitor ROM
    trigg
  APPLE2E.ROM: 33 KB
  
    0000.0FFF: C000.CFFF slot ROMs
    1000.3FFF: D000.FFFF Applesoft and Monitor ROM
    4000.4FFF: C000.CFFF built-in ROM overlay
    5000.7FFF: repeat of D000.FFFF ROM
*/

// Let's assume the IIe ROM for now
var CXXX_CARD_OFFSET = 0x0000;
var CXXX_ROM_OFFSET  = 0x4000;
var BASIC_ROM_OFFSET = 0x1000;

/** Read from slot ROMs */
function mem_get_Cxxx_card(addr) {
  return rom_data[CXXX_CARD_OFFSET + (addr - 0xC000)];
}

/** Read from built-in Cxxx ROMs */
function mem_get_Cxxx_ROM(addr) {
  return rom_data[CXXX_ROM_OFFSET + (addr - 0xC000)];
}

/** Read from Applesoft/Monitor ROM */
function mem_get_monitor_rom(addr) {
  return rom_data[BASIC_ROM_OFFSET + (addr - 0xD000)];
}

/** Read from Lang card Dxxx RAM bank 1 */
function mem_get_Dxxx_RAM_bank1(addr) {
  return raw_mem[addr - 0x1000]; // use the 0xC000.CFFF ram space
}

/** Read from Lang card Dxxx RAM bank 2 */
function mem_get_Dxxx_RAM_bank2(addr) {
  return raw_mem[addr]; // use the 0xD000.DFFF ram space
}

/** Read from Lang card Exxx-Fxxx RAM */
function mem_get_Exxx_RAM(addr) {
  return raw_mem[addr];
}

/** Write to high memory in the Dxxx bank */
function mem_set_Dxxx(addr, val) {
  if (lcWriteMode != "write1" && lcWriteMode != "write2")
    return;
  while(addr in blockMap[addr>>12])
    clearBlock(blockMap[addr>>12][addr][0]);
  if (lcWriteMode == "write1")
    raw_mem[addr - 0x1000] = val;
  else
    raw_mem[addr] = val;
}

/** Write to high memory in the Exxx or Fxxx banks */
function mem_set_Exxx(addr, val) {
  if (lcWriteMode != "write1" && lcWriteMode != "write2")
    return;
  while(addr in blockMap[addr>>12])
    clearBlock(blockMap[addr>>12][addr][0]);
  raw_mem[addr] = val;
}

/** Clear memory to power-on state */
function clearMem()
{
  raw_mem = new Array(0x10000);
  mem_set = new Array(0x10000);
  mem_run = new Array(0x10);
  blockMap = new Array(0x10);
  volatile1 = new Array(0x10);
  volatile2 = new Array(0x10);
  hibank_get = [];
  bankState = [];
  
  for (var i=0; i<=0xF; i++) {
    mem_run[i] = new Array(0x1000);
    blockMap[i] = [];
    volatile1[i] = [];
    volatile2[i] = [];
    if (i >= 0xC) {
      hibank_get[i] = new Array(0x1000);
      bankState[i] = [];
    }
  }

  CxxxMode = DxxxMode = ExxxMode = null;
  switchCxxxMode("card");
  switchDxxxMode("rom");
  switchExxxMode("rom");
  lcWriteMode = "protect";

  for (var i=0; i<0x10000; i += 0x100) {
    var setter = calc_setter(i); // same per page
    for (var j=0; j<0x100; j++) {
      raw_mem[i+j] = 0;
      mem_set[i+j] = setter;
      mem_run[i>>12][i+j] = recomp_run;
    }
  }
}

// Pad a number with spaces, for sorting purposes.
function pad(number, length) {
  var str = '' + number;
  while (str.length < length)
    str = ' ' + str;
  return str;
}

/** For testing, restart machine over and over */
function debugReboot()
{
  var totalComps = 0;
  var useRatios = [];
  var recompHotAddrs = [];
  for (var i in recompCounts) {
    recompHotAddrs.push([pad(recompCounts[i],10), parseInt(i)]);
    totalComps += recompCounts[i];
    var useRatio = parseInt(compRunCounts[i] / recompCounts[i]);
    if (!(useRatio in useRatios))
      useRatios[useRatio] = 0;
    ++useRatios[useRatio];
  }
  recompHotAddrs.sort();
  
  var nToPrint = 20;
  for (var i = recompHotAddrs.length - 1; i >= 0 && nToPrint > 0; i--, nToPrint--)
    console.log("Recomp hot: $" + toHex(recompHotAddrs[i][1]) + " recompiled " + recompHotAddrs[i][0] + " times");
    
  var totalRuns = 0;
  for (var i in compRunCounts)
    totalRuns += compRunCounts[i];
    
  console.log("Total comps: " + totalComps + ", total runs: " + totalRuns);
  
  for (var i in useRatios)
    console.log("Use ratio: " + i + " times for " + useRatios[i] + " funcs");
  
  var volatileHotAddrs = [];
  var volatileTotal = 0;
  for (var i in volatileCounts) {
    volatileHotAddrs.push([pad(volatileCounts[i],10), parseInt(i)]);
    volatileTotal += volatileCounts[i];
  }
  volatileHotAddrs.sort();
  
  var nToPrint = 20;
  for (var i = volatileHotAddrs.length - 1; i >= 0 && nToPrint > 0; i--, nToPrint--)
    console.log("Volatile hot: $" + toHex(volatileHotAddrs[i][1]) + " as " + volatileHotAddrs[i][0] + " cycles");
    
  console.log("Volatile total cycles: " + volatileTotal);
    
  assert(false); // force stop
  
  console.log("*** DEBUG REBOOT ***");
  clearMem();
  pc = 0xC600;
  rebootAfterTime = t + rebootInterval;
}

/** Calculate the hibank_get function pointers */
function calc_hibank_get(i, mode)
{
  assert(i >= 0xC000);
  if (i <= 0xC0FF)
    return mem_get_C0xx;
  else if (i <= 0xC2FF)
    return mode == "ROM" ? mem_get_Cxxx_ROM : mem_get_Cxxx_card;
  else if (i <= 0xC3FF)
    return mem_get_Cxxx_ROM; // kludge alert: we never have an external slot 3 rom, do we?
  else if (i <= 0xCFFF)
    return mode == "ROM" ? mem_get_Cxxx_ROM : mem_get_Cxxx_card;
  else if (i <= 0xDFFF)
    return mode == "ram1" ? mem_get_Dxxx_RAM_bank1 :
           mode == "ram2" ? mem_get_Dxxx_RAM_bank2 :
           mem_get_monitor_rom;
  else
    return mode == "ram" ? mem_get_Exxx_RAM : mem_get_monitor_rom;
}

function putBankState(bank, mode)
{
  if (!bank in bankState)
    bankState[bank] = [];
  if (!(mode in bankState[bank]))
    bankState[bank][mode] = [];
  bankState[bank][mode]['run'] = mem_run[bank];
  bankState[bank][mode]['volatile1'] = volatile1[bank];
  bankState[bank][mode]['volatile2'] = volatile2[bank];
  bankState[bank][mode]['blockMap'] = blockMap[bank];
  bankState[bank][mode]['hibank_get'] = hibank_get[bank];
}

function getBankState(bank, mode)
{
  if (bank in bankState && mode in bankState[bank]) {
    mem_run[bank]    = bankState[bank][mode]['run'];
    volatile1[bank]  = bankState[bank][mode]['volatile1'];
    volatile2[bank]  = bankState[bank][mode]['volatile2'];
    blockMap[bank]   = bankState[bank][mode]['blockMap'];
    hibank_get[bank] = bankState[bank][mode]['hibank_get'];
  }
  else {
    mem_run[bank]    = [];
    volatile1[bank]  = [];
    volatile2[bank]  = [];
    blockMap[bank]   = [];
    hibank_get[bank] = [];
    var base = bank << 12;
    var top = base + 0x1000;
    for (var i=base; i<top; i++) {
      mem_run[bank][i] = recomp_run;
      hibank_get[bank][i] = calc_hibank_get(i, mode);
    }
  }
}

function switchCxxxMode(newMode)
{
  if (newMode == CxxxMode)
    return;
  if (CxxxMode != null)
    putBankState(0xC, CxxxMode);
  getBankState(0xC, newMode);
  CxxxMode = newMode;
}

function switchDxxxMode(newMode)
{
  if (newMode == DxxxMode)
    return;
  if (DxxxMode != null)
    putBankState(0xD, DxxxMode);
  getBankState(0xD, newMode);
  DxxxMode = newMode;
}

function switchExxxMode(newMode)
{
  if (newMode == ExxxMode)
    return;
  if (ExxxMode != null) {
    putBankState(0xE, ExxxMode);
    putBankState(0xF, ExxxMode);
  }
  getBankState(0xE, newMode);
  getBankState(0xF, newMode);
  ExxxMode = newMode;
}

function setBlock(block)
{
  for (var i = block.start; i < block.end; i++) {
    if (!(i in blockMap[i>>12]))
      blockMap[i>>12][i] = new Array(0);
    blockMap[i>>12][i].push(block);
    if (i < 0xC000)
      mem_set[i] = uncomp_set; // we only do special uncomp tracking for non-bank-switched RAM
  }
}

function clearBlock(block)
{
  for (var i = block.start; i < block.end; i++) {
    if (blockMap[i>>12][i].length == 1) {
      delete blockMap[i>>12][i];
      mem_set[i] = calc_setter(i);
    }
    else
      blockMap[i>>12][i].splice(blockMap[i>>12][i].indexOf(block), 1);
  }
  mem_run[block.start>>12][block.start] = recomp_run;
}

var nComps = 0;

function isVolatile(addr) {
  return addr in volatile1[addr>>12] || addr in volatile2[addr>>12];
}

function markVolatile(addr) {
  volatile1[addr>>12][addr] = true;
  volatile1[(addr-1)>>12][addr-1] = true; // to account for 2-byte instrs
  volatile1[(addr-2)>>12][addr-2] = true; // to account for 3-byte instrs
}

function flushVolatility() 
{
  var area, i, j;
  
  // For addresses that are no longer volatile, allow recompilation.
  for (var area = 0; area <= 0xF; area++)
  {
    for (i in volatile2[area]) 
    {
      if (!(i in volatile1[area])) {
        if (mem_run[i>>12][i] == volatile_run) {
          mem_run[i>>12][i] = recomp_run;
          //console.debug("addr " + toHex(parseInt(i),4) + " no longer volatile.");
        }
      }
    }
  }
  
  // Now rotate.
  volatile2 = volatile1;
  volatile1 = new Array(0x10);
  for (var i = 0; i <= 0xF; i++)
    volatile1[i] = [];
}

function recomp_run() 
{
  var func;
  if (pc < 0x200)
    func = def_run;
  else if (isVolatile(pc))
    func = volatile_run;
  else 
  {
    var end = pc+3; 
    while (end < pc+66 && end < 0x10000 && !isVolatile(end))
      end++;
    var block = compiler.compile(pc, end-2);
    nComps++;
    //console.debug("\nFunction:\n" + block.code);
    try 
    {
      if (typeof block.code == "string")
        block.func = new Function(block.code);
      else
        block.func = block.code;
    }
    catch (e) {
      console.debug("Error compiling code: " + block.code);
    }
    func = block.func;
    setBlock(block);
  }
  
  lastCode = func;
  mem_run[pc>>12][pc] = func;
  func();
}

var recompCounts = new Array();
var compRunCounts = new Array();

function debugPC()
{
  var func = mem_run[pc>>12][pc];
  if (!("runCount" in func)) {
    func.runCount = 0;
    if (!(pc in recompCounts))
      recompCounts[pc] = 0;
    ++recompCounts[pc];
  }
  ++func.runCount;
  if (!(pc in compRunCounts))
    compRunCounts[pc] = 0;
  ++compRunCounts[pc];
  
  /*
  if (pc == 0xD0E5 && mem_get(pc+1) == 0xBD) {
    var addr = mem_get(0x44) + (mem_get(0x45)<<8);
    console.debug("ProDOS disk read to " + toHex(addr, 4) + ", err(C)=" + c + ":");
    var array = new Array();
    for (var i = 0; i < 256; i++)
      array[i] = mem_get(addr+i);
    hexDump(array);
  }
  */
  /*
  if (pc == 0xBD00)
    console.debug("RWTS: trk=" + raw_mem[0xB7EC] + ", sec=" + raw_mem[0xB7ED]);
  if (pc == 0xBDC7)
    console.debug("   RDADR: vol=" + raw_mem[0x2F] + ", trk=" + raw_mem[0x2E] + ", sec=" + raw_mem[0x2D] + ", cksum=" + raw_mem[0x2C] + ", C=" + c);
  */
}

function def_set(addr, val)
{
  raw_mem[addr] = val;
}

function readonly_set(addr, val)
{
  // yes, do nothing
}

function mem_set_C0xx(addr, val)
{
  // Simulate the 6502 write cycle, which first does a read.
  mem_get(addr);
  
  // Now handle write-only switches
  switch (addr)
  {
    case 0xC000: // CLR80STORE
      break;
    case 0xC001: // SET80STORE
      break;
    case 0xC002: // CLRAUXRD
      if (mem_get_saved != null) {
        mem_get = mem_get_saved;
        mem_get_saved = null;
      }
      break;
    case 0xC003: // SETAUXRD
      if (mem_get_saved == null) {
        mem_get_saved = mem_get;
        mem_get = function(addr) { 
          if (addr < 0x200 || addr > 0xC000)
            return mem_get_saved(addr);
          else
            return 0;
        };
      }
      break;
    case 0xC004: // CLRAUXWR
      // Inefficient hack
      if (mem_set_saved[0x200] !== undefined) {
        for (var i = 0x200; i < 0xC000; i++) {
          mem_set[i] = mem_set_saved[i];
          delete mem_set_saved[i];
        }
      }
      break;
    case 0xC005: // SETAUXWR
      // Inefficient hack
      if (mem_set_saved[0x200] === undefined) {
        function nullSet(addr, val) { }
        for (var i = 0x200; i < 0xC000; i++) {
          mem_set_saved[i] = mem_set[i];
          mem_set[i] = nullSet;
        }
      }
      break;
    case 0xC006: // CLRCXROM
      switchCxxxMode("card");
      break;
    case 0xC007: // SETCXROM
      switchCxxxMode("ROM");
      break;
    case 0xC008: // CLRAUXZP
      // Inefficient hack
      if (mem_set_saved[0] !== undefined) {
        for (var i = 0; i < 0x200; i++) {
          mem_set[i] = mem_set_saved[i];
          delete mem_set_saved[i];
        }
      }
      break;
    case 0xC009: // SETAUXZP
      // Inefficient hack
      if (mem_set_saved[0] === undefined) {
        function nullSet(addr, val) { }
        for (var i = 0; i < 0x200; i++) {
          mem_set_saved[i] = mem_set[i];
          mem_set[i] = nullSet;
        }
      }
      break;
    default:
      if (addr >= 0xC0E0 && addr <= 0xC0EF) // Disk II
        return diskDrive.softswitch_set(addr, val);
      break;
  }
}

/** 
 * Determine the memory change handler for the given address, based on
 * the current display mode.
 */
function calc_setter(addr)
{
  // Is the address on the current text or lo-res graphics page?  
  if (addr >= 0x400 && addr < 0x800 + (displayPage << 10)) {
    addr &= 0x7FF;
    var mode = (addr < 0x650) ? upperMode : lowerMode;
    if (mode == "text") return text_set;
    //if (mode == "gr") return gr_set;
  }
  
  // Is the address on the current hi-res graphics page?
  else if (addr >= 0x2000 && addr < 0x4000 + (displayPage << 13)) {
    addr &= 0x3FFF;
    var mode = ((addr & ~0x1C00) < 0x2250) ? upperMode : lowerMode;
    if (mode == "hgr") return hgr_set;
  }
  
  // Is it an I/O address?
  else if (addr >= 0xC000 && addr <= 0xC0FF)
    return mem_set_C0xx;
    
  // Is it a card ROM address? Then do a get instead.
  else if (addr >= 0xC100 && addr <= 0xCFFF)
    return mem_get;
    
  // Is it in language card Dxxx space?
  else if (addr >= 0xD000 && addr <= 0xDFFF)
    return mem_set_Dxxx;
    
  // Is it in the high language card space?
  else if (addr >= 0xE000)
    return mem_set_Exxx;
    
  // Regular old address.
  return def_set;
}

function uncomp_set(addr, val)
{
  if (raw_mem[addr] == val)
    return;
    
  assert(addr in blockMap[addr>>12]);
  while(addr in blockMap[addr>>12])
    clearBlock(blockMap[addr>>12][addr][0]);

  markVolatile(addr);
  
  var setFunc = mem_set[addr];
  assert(setFunc != uncomp_set);  
  setFunc(addr, val);
  
  mem_set[addr] = function(addr, val) {
    //console.debug("Still volatile: " + toHex(addr));
    markVolatile(addr);
    setFunc(addr, val);    
  }
}

/* ======================================================================= */

var a, x, y, v, c, nz, fd, fi, fb, pc, s, t, tmp, tmp2, mtmp1, mtmp2, mtmp3;
var branched;

/* ======================================================================= */

function toHex(val, nDigits)
{
  if (val === undefined)
    return "undefined";
  var ret = val.toString(16).toUpperCase();
  while (ret.length < nDigits)
    ret = "0" + ret;
  return ret;
}

function bool2(val) {
  if (val === undefined)
    return "undefined";
  if (val === 1 || val === true)
    return "1";
  if (val === 0 || val === false)
    return "0";
  return val;
}

var recentDebugs = [];

function debugRegs()
{
  recentDebugs.push([toHex(pc,4),"-   A=",toHex(a,2)," X=",toHex(x,2)," Y=",toHex(y,2),
                    " C=",bool2(c)," N=",bool2((nz & 0x80) ? 1 : 0)," Z=",bool2((nz & 0x200) || !nz)," V=",bool2(v),
                    " S=",toHex(s,2)," T=",t,
                    " stk=[", toHex(raw_mem[s+0x101],2), " ", 
                              toHex(raw_mem[s+0x102],2), " ",
                              toHex(raw_mem[s+0x103],2), " ",
                              toHex(raw_mem[s+0x104],2), "]"].join(""));
}

function printDebug()
{
  debugRegs();
  console.debug(recentDebugs.pop());
}

/** Dump a bunch of bytes to the console, 16 to a line for compactness */
function hexDump(array)
{
  var line = "";
  var base = 0;
  for (var i = 0; i < array.length; i++) {
    if ((i % 16) == 0)
      line += toHex(i, 4) + "- ";
    line += toHex(array[i], 2) + " ";
    if ((i % 8) == 7)
      line += "  ";
    if ((i % 16) == 15 || i == array.length-1) {
      line += "    ";
      for (var j = base; j <= i; j++) {
        if (array[j] >= 0x20 && array[j] <= 0x7F)
          line += String.fromCharCode(array[j]);
        else if (array[j] >= 0xA0 && array[j] <= 0xFF)
          line += String.fromCharCode(array[j] - 0x80);
        else
          line += ".";
      }
      console.debug(line);
      line = "";
      base = i+1
    }
  }
}
  
/* ======================================================================= */

var tickTimer = null;
var prevTime = null;

var targetFps = 20;
var targetHz = 1010000;
var cyclesPerTick = targetHz / targetFps;
var msecPerTick = 1000 / targetFps;

var cyclesDone = 0;

var debugCount = 0;

function addKey(keyChar)
{
  kbdBuf += keyChar;
  if (checker)
    checker.addKey(keyChar);
}

function keydown(event) 
{
  keyIsDown = true;
  
  // Sadly we don't get all of these (e.g. ESC) from bluetooth keyboard on iPhone.
  var ch = -1;
  switch (event.keyCode) {
    case 37: // left arrow
      ch = 8; break;
    case 38: // up arrow
      ch = 11; break;
    case 39: // right arrow
      ch = 21; break;
    case 40: // down arrow
      ch = 10; break;
    case 8:
    case 9:
    case 27:
      ch = event.keyCode; break;
  }
  if (ch >= 0) {
    addKey(String.fromCharCode(ch));
    return false;
  }
  return true;
}

function keyup(event)
{
  keyIsDown = false;
}

function keypress(event) 
{
  var ch = String.fromCharCode(event.which);
  if (softCapsLock && ch >= 'a' && ch <= 'z')
    ch = String.fromCharCode(event.which - 32);
  addKey(ch);
  return false;
}

var triggerTime = -1;
var triggerAddr = -1;
if (checker) {
  checker.setTriggerTime(triggerTime);
  checker.setTriggerAddr(triggerAddr);
}

function runCycles(end)
{
  if (checker) 
  {
    if (debugCount > 0) 
    {
      --debugCount;
      printDebug();
      var oldPC = pc;
      if (pc == 0xB9)
        console.debug("mem[11] = " + toHex(mem_get(11),2));
      mem_run[pc>>12][pc]();
      console.debug(mem_run[oldPC>>12][oldPC]);
      if (oldPC == 0xB9)
        console.debug("mem[11] = " + toHex(mem_get(11),2));
      if (checker)
        checker.runCycles(t);
      return;
    }
    
    while (t < end) {
      if (triggerTime >= 0 && t >= triggerTime && t < triggerTime+100) {
        triggerTime = 0;
        debugCount = 10;
        return;
      }
      if (pc == triggerAddr) {
        triggerAddr = 0;
        debugCount = 10;
        return;
      }
      last_run_t = t;
      mem_run[pc>>12][pc]();
      checker.runCycles(t);
    }
    return;
  }
  
  // Normal case
  while (t < end)
  {
    var littleEnd = (end-t > 100000) ? t+100000 : end;
    runCyclesLittle(littleEnd);
    
    // Every 30 seconds reset the volatility index
    if (t < lastVolatilityFlush || t - lastVolatilityFlush > 30000000) {
      flushVolatility();
      lastVolatilityFlush = t;
    }
  }
}

function runCyclesLittle(littleEnd)
{
  while (t < littleEnd)
    mem_run[pc>>12][pc]();
}

var prevTickTime = new Date().getTime();
var endTime;

function tick()
{
  //try {
  
    var tstart = t;
    var start = (new Date).getTime();
    var msecSincePrevTick = start - prevTickTime;
    prevTickTime = start;
    
    if (kbdPause) {
      --kbdPause;
      if (kbdPause)
        return;
      else
        x = y = 1; // simulate normal kbd loop termination
    }

    if (ballsOut) {
      if (msecSincePrevTick < (msecPerTick * 1.1))
        cyclesPerTick = Math.round(cyclesPerTick * 1.1);
      else if (msecSincePrevTick > (msecPerTick * 0.9))
        cyclesPerTick = Math.round(cyclesPerTick * 0.9);
    }
    else {
      cyclesPerTick = targetHz / targetFps;
      var ratio = msecSincePrevTick * (1.0 / msecPerTick);
      if (ratio > 1.2)
        ratio = 1.2;
      cyclesPerTick = Math.round(cyclesPerTick * ratio);
    }
  
    endTime = t + cyclesPerTick; // needed by compiled while loop, and mem_gets
    runCycles(endTime);
    refreshHires();
    ++framesDone;
    cyclesDone += (t - tstart);
  /*}
  catch (err) {
    stop6502();
    setEmuStatus(err);
    throw err;
  }*/
  
  if (rebootInterval >= 0 && t >= rebootAfterTime)
    debugReboot();
}

var fpsTimer = null;
var framesDone = 0;

function updateFps()
{
  var now = (new Date).getTime();
  var diff = now - startTime;
  var cpuRatio = Math.round(cyclesDone / diff / 10) / 100.0;
  var fps = framesDone / (diff / 1000.0);
  setEmuStatus("speed: " + cpuRatio.toFixed(2) + ", fps=" + fps.toFixed(2),
    "cyclesPerTick=" + cyclesPerTick + ", nComps=" + nComps);
  startTime = now;
  cyclesDone = 0;
  framesDone = 0;
}

/* ======================================================================= */

var compiler;
var startTime;

function insertDisk(diskData)
{
  diskDrive.insert(diskData);
  if (checker)
    checker.insertDisk(diskData);
}

function ejectDisk()
{
  return diskDrive.eject();
}

function init6502(canv)
{
  // Get an instance of the compiler, attached to this memory.
  compiler = new Compiler(mem_get);
  
  // Create a disk drive
  diskDrive = new DiskDrive();
  
  // Record the canvas for permanent use, and clear it
  canvas = canv;
  
  // Clear the canvas to black
  mainCtx = canvas.getContext('2d');
  mainCtx.save();
  mainCtx.fillStyle = "#000000";
  mainCtx.fillRect (0, 0, 280, 192);
  mainCtx.restore();

  // Init the display modes
  textInit();
  hgrInit();

  // Make sure we get keystrokes  
  document.getElementById('keyboard').onkeypress = keypress;
  document.getElementById('keyboard').onkeydown = keydown;
  document.getElementById('keyboard').onkeyup = keyup;
  document.getElementById('keyboard').focus(); //this pops up keyboard on iPhone, not desirable
}

function setCpuSpeed(mhz)
{
  targetHz = mhz * 1000000;
  cyclesPerTick = targetHz / targetFps;
  msecPerTick = 1000 / targetFps;
}

function setAppleRom(data)
{
  rom_data = new Array();
  for (var i in data)
    rom_data[i] = data.charCodeAt(i) & 0xFF;
}

function run6502(startAddr)
{
  // If we're running, stop it.
  stop6502();
  
  // Init registers and memory
  a = x = y = c = v = nz = t = 0;
  s = 0xFF;
  clearMem();
  
  if (startAddr !== undefined)
    pc = startAddr;
  else
    pc = mem_get(0xFFFC) + (mem_get(0xFFFD) << 8); // Reset vector
      
  // Init checker if we have one
  if (checker)
    checker.init(pc);
    
  startTime = (new Date).getTime();
  resume6502();
}

function reset6502()
{
  pc = mem_get(0xFFFC) + (mem_get(0xFFFD) << 8); // Reset vector
}

function pause6502()
{
  if (fpsTimer) {
    clearInterval(fpsTimer);
    fpsTimer = null;
  }
  if (tickTimer) {
    clearInterval(tickTimer);
    tickTimer = null;
  }
}

function resume6502()
{
  if (fpsTimer == null)
    fpsTimer = setInterval(updateFps, 2000);
  if (tickTimer == null)
    tickTimer = setInterval(tick, msecPerTick);
}

function stop6502()
{
  pause6502();
}
